
#ifndef INTER_PROCESSOR_REQUEST_COMMON_H
#define INTER_PROCESSOR_REQUEST_COMMON_H

#include "iprq_cfg.h"
#include "iprq.h"

#include <asm/io.h> /* inl, outl */
#include <linux/threads.h> /* NR_CPUS */
#include <linux/preempt.h> /* preempt_disable, preempt_enable */
#include <linux/kernel.h> /* printk */
#include <dtvrecdd/memtrans.h>

#ifndef NULL
#define NULL	((void *)0)
#endif

#define MB()	\
do { \
	volatile unsigned int tmp; \
	asm volatile ("dmb": : :"memory"); \
	tmp = *((volatile unsigned int *)iprq_gxicr_base0); \
} while(0)

#define PI()	asm volatile ("bkpt": : :"memory")

static inline int MEMCPY(void *dst, void *src, unsigned int size)
{
	unsigned char *p = dst, *q = src;
	unsigned int n = size;
	if ((((unsigned int)dst | (unsigned int)src) & 0x3) == 0) {
		while (n >= 4) {
			*(unsigned int *)p = *(unsigned int *)q;
			p += 4;
			q += 4;
			n -= 4;
		}
	}
	while (n) {
		*p = *q;
		p++;
		q++;
		n--;
	}
	return 0;
}

static inline int MEMSET(void *dst, unsigned char pat, unsigned int size)
{
	unsigned char *p = dst;
	unsigned int n = size, pat4;
	if (((unsigned int)dst & 0x3) == 0) {
		pat4 = (pat << 24) | (pat << 16) | (pat << 8) | pat;
		while (n >= 4) {
			*(unsigned int *)p = pat4;
			p += 4;
			n -= 4;
		}
	}
	while (n) {
		*p = pat;
		p++;
		n--;
	}
	return 0;
}

#define __GXICR_BASE0	0x50700000
#define __GXICR_BASE1	0x50800000
#define __GXICR(x)	(__GXICR_BASE0 + (x) * 4 + ((x) >= 512) * 0xff800)

#define __GXICR_DETECT	0x00000001
#define __GXICR_REQUEST	0x00000010
#define __GXICR_ENABLE	0x00000100
#define __GXICR_LEVEL	0x0000f000
#define __GXICR_LPINT	0x00020000

#define __CPUINT_BASE	(0x50801000)
#define __CPUINT	(0x50801000)
#define __CPUINTSET	(0x50801004)
#define __CPUINTCLR	(0x50801008)

extern unsigned long iprq_gxicr_base0;
extern unsigned long iprq_gxicr_base1;
extern unsigned long iprq_cpuint_base;

static inline void __iomem *__iprq_io(unsigned long addr)
{
	switch (addr) {
	case __GXICR_BASE0 ... (__GXICR_BASE0 + 0x1000 - 1):
		return (void __iomem *)(addr - __GXICR_BASE0 + iprq_gxicr_base0);
	case __GXICR_BASE1 ... (__GXICR_BASE1 + 0x1000 - 1):
		return (void __iomem *)(addr - __GXICR_BASE1 + iprq_gxicr_base1);
	case __CPUINT_BASE ... (__CPUINT_BASE + 0x1000 - 1):
		return (void __iomem *)(addr - __CPUINT_BASE + iprq_cpuint_base);
	default:
		printk(KERN_ERR "__iprq_io: Unmapped address(%08lx)\n", addr);
		BUG();
	}
}

static inline unsigned long __INL(unsigned long addr)
{
	return readl(__iprq_io(addr));
}

static inline void __OUTL(unsigned long b, unsigned long addr)
{
	writel(b, __iprq_io(addr));
}

#define __LOCAL_SAVE_FLAGS(x) \
do {						\
	__asm__ __volatile__(			\
		"	mrs	%0, cpsr\n"	\
		: "=r"(x)			\
		:				\
		: "memory", "cc"		\
		);				\
} while (0)

#define __LOCAL_CLI()				\
do {						\
	__asm__ __volatile__(			\
		"	cpsid	i	\n"	\
		:				\
		:				\
		: "memory", "cc"		\
		);				\
	MB();					\
} while (0)

#define __LOCAL_CLI_SAVE(x)	\
do {				\
	__LOCAL_SAVE_FLAGS(x);	\
	__LOCAL_CLI();		\
} while (0)

#define __LOCAL_IRQ_RESTORE(x)			\
do {						\
	MB();					\
	__asm__ __volatile__(			\
		"	msr	cpsr_c, %0\n"	\
		:				\
		: "r"(x)			\
		: "memory", "cc"		\
		);				\
} while (0)

static inline void LOCK(volatile int *lock)
{
	unsigned long tmp;

#ifdef IPRQ_USE_SWP
	__asm__ __volatile__(
		"1:	swp	%0, %2, [%1]\n"
		"	cmp	%0, #0\n"
		"	bne	1b"
		: "=&r" (tmp)
		: "r" (lock), "r" (1)
		: "cc", "memory");
#else
	__asm__ __volatile__(
		"1:	ldrex	%0, [%1]\n"
		"	teq	%0, #0\n"
		"	strexeq	%0, %2, [%1]\n"
		"	teqeq	%0, #0\n"
		"	bne	1b"
		: "=&r" (tmp)
		: "r" (lock), "r" (1)
		: "cc", "memory");
#endif
	MB();
}

static inline void UNLOCK(volatile int *lock)
{
	MB();
#ifdef IPRQ_ASSERT
	if (*lock != 1) {
		IPRQ_ABORT();
	}
#endif
	__asm__ __volatile__(
		"	str	%1, [%0]\n"
		"	dsb"
		:
		: "r" (lock), "r" (0)
		: "cc", "memory");
}

static inline int TRYLOCK(volatile int *lock)
{
	unsigned long tmp;

#ifdef IPRQ_USE_SWP
	__asm__ __volatile__(
		"	swp	%0, %2, [%1]\n"
		: "=&r" (tmp)
		: "r"(lock), "r" (1)
		: "cc", "memory");
	MB();
	return tmp ^ 1;
#else
	__asm__ __volatile__(
		"	ldrex	%0, [%1]\n"
		"	teq	%0, #0\n"
		"	strexeq	%0, %2, [%1]"
		: "=&r" (tmp)
		: "r"(lock), "r" (1)
		: "cc", "memory");
	if (!tmp){
		MB();
		return 1;
	} else {
		return 0;
	}
#endif
}

static inline unsigned long LOCK_CLI_SAVE(volatile int *lock)
{
	unsigned long flags;

	__LOCAL_CLI_SAVE(flags);
	while (!TRYLOCK(lock)) {
		__LOCAL_IRQ_RESTORE(flags);
		__LOCAL_CLI_SAVE(flags);
	}
	return flags;
}


static inline void UNLOCK_IRQ_RESTORE(volatile int *lock, unsigned long flags)
{
	UNLOCK(lock);
	__LOCAL_IRQ_RESTORE(flags);
}

#define __DSP_INIT(x)		int x __attribute__((unused))
#define __DSP_SAVE(x)		preempt_disable()
#define __DSP_RESTORE(x)	preempt_enable()

#if NR_CPUS >= 2
#define DEFINE_LOCAL_LOCK(lock)	volatile int lock = 0
#define LOCAL_LOCK(lock, x)	\
do {				\
	__DSP_SAVE(x);		\
	LOCK(lock);		\
} while (0)
#define LOCAL_UNLOCK(lock, x)	\
do {				\
	UNLOCK(lock);		\
	__DSP_RESTORE(x);	\
} while (0)
#else
#define DEFINE_LOCAL_LOCK(lock)	int lock __attribute__((unused))
#define LOCAL_LOCK(lock, x)	__DSP_SAVE(x)
#define LOCAL_UNLOCK(lock, x)	__DSP_RESTORE(x)
#endif

static inline void __MASK_AND_SET_ICR(unsigned int irq,
				      unsigned int mask, unsigned int set)
{
	unsigned long tmp32, flags;
	__LOCAL_CLI_SAVE(flags);

	tmp32 = __INL(__GXICR(irq));
	tmp32 = (tmp32 & mask) | set;
	__OUTL(tmp32, __GXICR(irq));
	tmp32 = __INL(__GXICR(irq));

	__LOCAL_IRQ_RESTORE(flags);
}

#define IRQ_ENABLE(irq)		\
	__MASK_AND_SET_ICR(irq, ~(__GXICR_REQUEST | __GXICR_DETECT), __GXICR_ENABLE)
#define IRQ_DISABLE(irq)	\
	__MASK_AND_SET_ICR(irq, ~(__GXICR_ENABLE | __GXICR_REQUEST | __GXICR_DETECT), 0)
#define IRQ_CLEAR(irq)		\
	__MASK_AND_SET_ICR(irq, ~__GXICR_REQUEST, __GXICR_DETECT)
#define IRQ_REQUEST(irq)	\
	__MASK_AND_SET_ICR(irq, ~0, __GXICR_REQUEST | __GXICR_DETECT)
#define IRQ_SETLPINT(irq)	\
	__MASK_AND_SET_ICR(irq, ~0, __GXICR_LPINT)
#define NMI_ENABLE(irq)		\
	__MASK_AND_SET_ICR(irq, ~(__GXICR_REQUEST | __GXICR_DETECT), __GXICR_ENABLE)
#define NMI_REQUEST(irq)	\
	__MASK_AND_SET_ICR(irq, ~0, __GXICR_REQUEST | __GXICR_DETECT)


#define IS_IRQ_ENABLE(irq)	(__INL(__GXICR(irq)) & __GXICR_ENABLE)
#define IS_IRQ_DETECTED(irq)	(__INL(__GXICR(irq)) & __GXICR_DETECT)

#define SET_CPUINT(n)	(__OUTL(1 << (n), __CPUINTSET))
#define CLR_CPUINT(n)	(__OUTL(1 << (n), __CPUINTCLR))

static inline int GET_IPRQ_IRQ(int osid) {
	if (osid == OS_R) {
		return IPRQ_L_TO_R_IRQ;
	} else if (osid == OS_L) {
		return IPRQ_R_TO_L_IRQ;
	}
	IPRQ_ABORT();
}

static inline int GET_IPRQ_ARMGIC(int osid) {
	if (osid == OS_L) {
		return IPRQ_R_TO_L_GIC;
	}
	IPRQ_ABORT();
}

#define DDR_DSP_STATUS_BASE		(DDR_CH0_START + 0x0000c000)
#define DDR_DSP_STATUS_SIZE		0x00004000
#define DDR_DSP_STATUS_OFFSET_REG	0x0400
#define DDR_DSP_STATUS_OFFSET_R0	(DDR_DSP_STATUS_OFFSET_REG + 0x0000)
#define DDR_DSP_STATUS_OFFSET_R1	(DDR_DSP_STATUS_OFFSET_REG + 0x0004)
#define DDR_DSP_STATUS_OFFSET_R2	(DDR_DSP_STATUS_OFFSET_REG + 0x0008)
#define DDR_DSP_STATUS_OFFSET_R3	(DDR_DSP_STATUS_OFFSET_REG + 0x000c)
#define DDR_DSP_STATUS_OFFSET_R4	(DDR_DSP_STATUS_OFFSET_REG + 0x0010)
#define DDR_DSP_STATUS_OFFSET_R5	(DDR_DSP_STATUS_OFFSET_REG + 0x0014)
#define DDR_DSP_STATUS_OFFSET_R6	(DDR_DSP_STATUS_OFFSET_REG + 0x0018)
#define DDR_DSP_STATUS_OFFSET_R7	(DDR_DSP_STATUS_OFFSET_REG + 0x001c)
#define DDR_DSP_STATUS_OFFSET_R8	(DDR_DSP_STATUS_OFFSET_REG + 0x0020)
#define DDR_DSP_STATUS_OFFSET_R9	(DDR_DSP_STATUS_OFFSET_REG + 0x0024)
#define DDR_DSP_STATUS_OFFSET_R10	(DDR_DSP_STATUS_OFFSET_REG + 0x0028)
#define DDR_DSP_STATUS_OFFSET_R11	(DDR_DSP_STATUS_OFFSET_REG + 0x002c)
#define DDR_DSP_STATUS_OFFSET_R12	(DDR_DSP_STATUS_OFFSET_REG + 0x0030)
#define DDR_DSP_STATUS_OFFSET_R13	(DDR_DSP_STATUS_OFFSET_REG + 0x0034)
#define DDR_DSP_STATUS_OFFSET_R14	(DDR_DSP_STATUS_OFFSET_REG + 0x0038)
#define DDR_DSP_STATUS_OFFSET_R15	(DDR_DSP_STATUS_OFFSET_REG + 0x003c)
#define DDR_DSP_STATUS_OFFSET_R16	(DDR_DSP_STATUS_OFFSET_REG + 0x0040)
#define DDR_DSP_STATUS_OFFSET_R17	(DDR_DSP_STATUS_OFFSET_REG + 0x0044)
#define DDR_DSP_STATUS_OFFSET_R18	(DDR_DSP_STATUS_OFFSET_REG + 0x0048)
#define DDR_DSP_STATUS_OFFSET_R19	(DDR_DSP_STATUS_OFFSET_REG + 0x004c)
#define DDR_DSP_STATUS_OFFSET_R20	(DDR_DSP_STATUS_OFFSET_REG + 0x0050)
#define DDR_DSP_STATUS_OFFSET_R21	(DDR_DSP_STATUS_OFFSET_REG + 0x0054)
#define DDR_DSP_STATUS_OFFSET_R22	(DDR_DSP_STATUS_OFFSET_REG + 0x0058)
#define DDR_DSP_STATUS_OFFSET_R23	(DDR_DSP_STATUS_OFFSET_REG + 0x005c)
#define DDR_DSP_STATUS_OFFSET_R24	(DDR_DSP_STATUS_OFFSET_REG + 0x0060)
#define DDR_DSP_STATUS_OFFSET_R25	(DDR_DSP_STATUS_OFFSET_REG + 0x0064)
#define DDR_DSP_STATUS_OFFSET_R26	(DDR_DSP_STATUS_OFFSET_REG + 0x0068)
#define DDR_DSP_STATUS_OFFSET_R27	(DDR_DSP_STATUS_OFFSET_REG + 0x006c)
#define DDR_DSP_STATUS_OFFSET_R28	(DDR_DSP_STATUS_OFFSET_REG + 0x0070)
#define DDR_DSP_STATUS_OFFSET_R29	(DDR_DSP_STATUS_OFFSET_REG + 0x0074)
#define DDR_DSP_STATUS_OFFSET_R30	(DDR_DSP_STATUS_OFFSET_REG + 0x0078)
#define DDR_DSP_STATUS_OFFSET_R31	(DDR_DSP_STATUS_OFFSET_REG + 0x007c)
#define DDR_DSP_STATUS_OFFSET_PSR	(DDR_DSP_STATUS_OFFSET_REG + 0x1000)
#define DDR_DSP_STATUS_OFFSET_PC	(DDR_DSP_STATUS_OFFSET_REG + 0x1008)
#define DDR_DSP_STATUS_OFFSET_LR	(DDR_DSP_STATUS_OFFSET_REG + 0x1018)
#define DDR_DSP_STATUS_OFFSET_ESB	(DDR_DSP_STATUS_OFFSET_REG + 0x1050)

#endif /* INTER_PROCESSOR_REQUEST_COMMON_H */
